<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/auth_check.php';
require_once '../includes/functions.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';

// Check if user has permission to create QR codes
if (!hasPermission('create_qr_code')) {
    header('Location: dashboard.php');
    exit;
}

$activePage = 'qrcodes';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Get all products with their serial numbers
$stmt = $pdo->query("
    SELECT p.id, p.name, p.quantity, p.price, 
           GROUP_CONCAT(psn.serial_number) as serial_numbers
    FROM products p
    LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id AND psn.status = 'available'
    GROUP BY p.id
    ORDER BY p.name
");
$products = $stmt->fetchAll();

// Handle form submission
$selected_id = isset($_GET['product_id']) ? intval($_GET['product_id']) : 0;
$print_count = isset($_GET['print_count']) ? max(1, intval($_GET['print_count'])) : 1;
$qr_server = isset($_GET['qr_server']) ? $_GET['qr_server'] : 'qrserver';
$filtered_product = null;
if ($selected_id) {
    foreach ($products as $p) {
        if ($p['id'] == $selected_id) {
            $filtered_product = $p;
            break;
        }
    }
}

// Function to generate QR code data URL
function generateQRCodeDataUrl($data, $productInfo = null) {
    // Always use just the serial number or product name (no JSON structure)
    $qrData = $data;
    
    // Use QR Server API for QR code generation (more reliable and customizable)
    $size = 200; // QR code size in pixels
    $margin = 4; // Margin around QR code
    $errorCorrection = 'M'; // Medium error correction level
    $format = 'png'; // Image format
    $url = "https://api.qrserver.com/v1/create-qr-code/?size={$size}x{$size}&data=" . urlencode($qrData) . "&margin={$margin}&ecc={$errorCorrection}&format={$format}";
    return $url;
}

// Function to generate QR code using alternative server (fallback)
function generateAlternativeQRCode($data, $productInfo = null) {
    // Always use just the serial number or product name (no JSON structure)
    $qrData = $data;
    
    // Use QR Code Monkey API as alternative (more customization options)
    $size = 200;
    $url = "https://www.qrcode-monkey.com/api/qr/custom?size={$size}&data=" . urlencode($qrData) . "&file=png&download=0";
    return $url;
}

// Function to generate QR code using GoQR.me API
function generateGoQRCode($data, $productInfo = null) {
    // Always use just the serial number or product name (no JSON structure)
    $qrData = $data;
    
    // Use GoQR.me API (simple and reliable)
    $size = 200;
    $url = "https://api.qrserver.com/v1/create-qr-code/?size={$size}x{$size}&data=" . urlencode($qrData);
    return $url;
}

// Function to generate QR code using local library (alternative)
function generateLocalQRCode($data) {
    // Check if we have the TCPDF QR code library available
    if (class_exists('TCPDF2DBarcode')) {
        try {
            $barcode = new TCPDF2DBarcode($data, 'QRCODE,H');
            $qr_array = $barcode->getBarcodeArray();
            
            // Convert array to image data URL
            // This would require additional image processing
            return generateQRCodeDataUrl($data); // Fallback to QR Server API
        } catch (Exception $e) {
            return generateQRCodeDataUrl($data); // Fallback to QR Server API
        }
    } else {
        return generateQRCodeDataUrl($data); // Use QR Server API as default
    }
}

// Function to generate QR code based on selected server
function generateQRCode($data, $productInfo = null, $server = 'qrserver') {
    switch ($server) {
        case 'qrcodemonkey':
            return generateAlternativeQRCode($data, $productInfo);
        case 'goqr':
            return generateGoQRCode($data, $productInfo);
        case 'qrserver':
        default:
            return generateQRCodeDataUrl($data, $productInfo);
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>QR Codes - POS System</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body { background: #f8fafc; font-family: 'Inter', sans-serif; }
        .barcode-grid {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 0.3cm;
            justify-items: center;
            padding: 0.2cm;
        }
        .barcode-label {
            border: 1px solid #bbb;
            border-radius: 6px;
            background: #fff;
            width: 4.5cm;
            min-height: 4.2cm;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: flex-start;
            margin: 0.1cm;
            padding: 0.15cm 0.1cm 0.1cm 0.1cm;
            box-shadow: 0 1px 4px rgba(0,0,0,0.07);
            font-family: 'Inter', Arial, sans-serif;
        }
        .barcode-title {
            font-weight: 700;
            font-size: 0.9rem;
            margin-bottom: 0.08cm;
            color: #222;
            text-align: center;
            width: 100%;
            border-bottom: 1px solid #eee;
            padding-bottom: 0.05cm;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
            line-height: 1.2;
        }
        .barcode-img {
            width: 2.5cm;
            height: 2.5cm;
            margin-bottom: 0.06cm;
            background: #fff;
            padding: 0.05cm;
            object-fit: contain;
            border: 1px solid #eee;
        }
        .barcode-serial, .barcode-price {
            font-size: 0.8rem;
            color: #333;
            margin-bottom: 0.04cm;
            text-align: center;
            width: 100%;
            word-break: break-word;
            line-height: 1.2;
        }
        .barcode-price {
            font-weight: 600;
            color: #1a4;
        }
        .barcode-actions { 
            margin-top: 0.2cm; 
        }
        @media print {
            body { 
                background: #fff !important;
                -webkit-print-color-adjust: exact !important;
                print-color-adjust: exact !important;
            }
            .no-print { display: none !important; }
            .container-fluid, .row, .col-md-10, .col-sm-6, .mb-4, .px-md-4 { 
                width: auto !important; 
                padding: 0 !important; 
                margin: 0 !important; 
            }
            .barcode-grid {
                display: grid !important;
                grid-template-columns: repeat(4, 1fr) !important;
                gap: 0.3cm !important;
                justify-items: center !important;
                padding: 0.2cm !important;
                background: none !important;
            }
            .barcode-label {
                page-break-inside: avoid;
                box-shadow: none !important;
                border: 1px solid #000 !important;
                margin: 0.1cm !important;
                width: 4.5cm !important;
                background: none !important;
                padding: 0.1cm !important;
            }
            .barcode-title {
                color: #000 !important;
                border-bottom: 1px solid #000 !important;
            }
            .barcode-serial, .barcode-price {
                color: #000 !important;
            }
            .barcode-img {
                background: none !important;
                padding: 0 !important;
            }
            .barcode-actions { display: none !important; }
            /* Hide floating circular/FAB/menu buttons from printouts */
            button[style*="border-radius: 50%"],
            .fab,
            .fab-print,
            .menu-btn,
            .floating-btn,
            .circle,
            .btn-circle {
                display: none !important;
            }
        }

        /* Add responsive styles */
        @media (max-width: 768px) {
            .container-fluid {
                padding: 10px;
            }

            .row {
                margin: 0;
            }

            .col-md-3, .col-md-9 {
                padding: 5px;
            }

            .card {
                margin-bottom: 10px;
            }

            .card-body {
                padding: 15px;
            }

            /* QR Code Grid */
            .qr-grid {
                display: grid;
                grid-template-columns: repeat(2, 1fr);
                gap: 10px;
                padding: 10px;
            }

            .qr-item {
                padding: 10px;
                margin: 0;
            }

            .qr-code {
                width: 100%;
                height: 70px;
                max-width: 150px;
                margin: 0 auto;
            }

            .qr-details {
                font-size: 0.9rem;
            }

            .qr-title {
                font-size: 1rem;
                margin-bottom: 5px;
            }

            .qr-price {
                font-size: 0.9rem;
            }

            /* Search and Filter Section */
            .search-section {
                flex-direction: column;
                gap: 10px;
            }

            .search-input {
                width: 100%;
            }

            .filter-select {
                width: 100%;
            }

            /* Print Layout */
            @media print {
                .qr-grid {
                    grid-template-columns: repeat(3, 1fr);
                    gap: 15px;
                    padding: 15px;
                }

                .qr-item {
                    page-break-inside: avoid;
                }

                .qr-code {
                    max-width: 120px;
                }

                .qr-details {
                    font-size: 0.8rem;
                }
            }

            .barcode-grid {
                grid-template-columns: repeat(2, 1fr);
                gap: 0.15cm;
            }
            .barcode-label {
                width: 100%;
                max-width: 4.5cm;
            }
        }

        /* Additional breakpoint for very small screens */
        @media (max-width: 480px) {
            .qr-grid {
                grid-template-columns: 1fr;
            }

            .qr-item {
                padding: 8px;
            }

            .qr-code {
                max-width: 130px;
            }

            .qr-details {
                font-size: 0.85rem;
            }

            .qr-title {
                font-size: 0.95rem;
            }

            .qr-price {
                font-size: 0.85rem;
            }

            /* Adjust buttons for small screens */
            .btn {
                padding: 6px 10px;
                font-size: 0.9rem;
            }

            .btn-sm {
                padding: 4px 8px;
                font-size: 0.8rem;
            }

            /* Adjust search and filter inputs */
            .form-control {
                font-size: 0.9rem;
                padding: 6px 10px;
            }

            .form-select {
                font-size: 0.9rem;
                padding: 6px 10px;
            }
        }

        /* Tablet-specific adjustments */
        @media (min-width: 769px) and (max-width: 1024px) {
            .qr-grid {
                grid-template-columns: repeat(3, 1fr);
                gap: 15px;
            }

            .qr-code {
                max-width: 180px;
            }

            .qr-details {
                font-size: 0.95rem;
            }
        }

        /* Ensure sidebar doesn't overlap */
        @media (max-width: 768px) {
            .sidebar {
                position: fixed;
                top: 0;
                left: 0;
                height: 100%;
                z-index: 1000;
                transform: translateX(-100%);
                transition: transform 0.3s ease-in-out;
            }

            .sidebar.show {
                transform: translateX(0);
            }

            .main-content {
                margin-left: 0;
                width: 100%;
            }

            /* Add overlay when sidebar is open */
            .sidebar-overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(0, 0, 0, 0.5);
                z-index: 999;
            }

            .sidebar-overlay.show {
                display: block;
            }
        }
    </style>
</head>
<body>
<div class="container-fluid py-4">
    <div class="row">
        <?php include '../includes/sidebar.php'; ?>
        <div class="col-md-10 ms-sm-auto px-md-4">
            <div class="no-print">
                <?php include '../includes/topbar.php'; ?>
            </div>
            <div class="container-fluid mt-4">
                <div class="d-flex justify-content-between align-items-center mb-4 no-print">
                    <h2 class="mb-0">Generate QR Codes for Products</h2>
                </div>
                <form class="row g-3 align-items-end mb-4 no-print" method="get" action="">
                    <div class="col-md-4">
                        <label class="form-label">Select Product</label>
                        <select name="product_id" class="form-select" required>
                            <option value="">-- Select Product --</option>
                            <?php foreach ($products as $product): ?>
                                <option value="<?php echo $product['id']; ?>" <?php if ($selected_id == $product['id']) echo 'selected'; ?>>
                                    <?php echo htmlspecialchars($product['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Number of Labels</label>
                        <input type="number" name="print_count" class="form-control" min="1" max="100" value="<?php echo $print_count; ?>" required>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">QR Server</label>
                        <select name="qr_server" class="form-select">
                            <option value="qrserver" <?php echo (isset($_GET['qr_server']) && $_GET['qr_server'] == 'qrserver') ? 'selected' : 'selected'; ?>>QR Server (Default)</option>
                            <option value="qrcodemonkey" <?php echo (isset($_GET['qr_server']) && $_GET['qr_server'] == 'qrcodemonkey') ? 'selected' : ''; ?>>QR Code Monkey</option>
                            <option value="goqr" <?php echo (isset($_GET['qr_server']) && $_GET['qr_server'] == 'goqr') ? 'selected' : ''; ?>>GoQR.me</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <button type="submit" class="btn btn-primary w-100"><i class="fas fa-qrcode me-1"></i>Show QR Codes</button>
                    </div>
                    <div class="col-md-2">
                        <a href="qrcodes.php" class="btn btn-outline-secondary w-100">Show All</a>
                    </div>
                </form>
                <div class="mb-3 no-print">
                    <button class="btn btn-success" onclick="window.print()"><i class="fas fa-print me-1"></i>Print All</button>
                </div>
                <div class="barcode-grid">
                <?php if ($filtered_product): ?>
                    <?php if (!empty($filtered_product['serial_numbers'])): ?>
                        <?php 
                        $serial_numbers = explode(',', $filtered_product['serial_numbers']);
                        foreach ($serial_numbers as $serial): 
                        ?>
                        <div class="barcode-label">
                            <div class="barcode-title"><?php echo htmlspecialchars($filtered_product['name']); ?></div>
                            <img src="<?php echo generateQRCode(trim($serial), null, $qr_server); ?>"
                                 alt="QR Code"
                                 class="barcode-img">
                            <div class="barcode-serial">S/N: <?php echo htmlspecialchars(trim($serial)); ?></div>
                            <div class="barcode-price">Price: <?php echo number_format($filtered_product['price'], 2); ?></div>
                        </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div class="barcode-label">
                            <div class="barcode-title"><?php echo htmlspecialchars($filtered_product['name']); ?></div>
                            <img src="<?php echo generateQRCode($filtered_product['name'], null, $qr_server); ?>"
                                 alt="QR Code"
                                 class="barcode-img">
                            <div class="barcode-price">Price: <?php echo number_format($filtered_product['price'], 2); ?></div>
                        </div>
                    <?php endif; ?>
                <?php else: ?>
                    <?php foreach ($products as $product): ?>
                        <?php if (!empty($product['serial_numbers'])): ?>
                            <?php 
                            $serial_numbers = explode(',', $product['serial_numbers']);
                            foreach ($serial_numbers as $serial): 
                            ?>
                            <div class="barcode-label">
                                <div class="barcode-title"><?php echo htmlspecialchars($product['name']); ?></div>
                                <img src="<?php echo generateQRCode(trim($serial), null, $qr_server); ?>"
                                     alt="QR Code"
                                     class="barcode-img">
                                <div class="barcode-serial">S/N: <?php echo htmlspecialchars(trim($serial)); ?></div>
                                <div class="barcode-price">Price: <?php echo number_format($product['price'], 2); ?></div>
                            </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <div class="barcode-label">
                                <div class="barcode-title"><?php echo htmlspecialchars($product['name']); ?></div>
                                <img src="<?php echo generateQRCode($product['name'], null, $qr_server); ?>"
                                     alt="QR Code"
                                     class="barcode-img">
                                <div class="barcode-price">Price: <?php echo number_format($product['price'], 2); ?></div>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
function printQRCode(qrCodeId) {
    var img = document.getElementById(qrCodeId);
    var win = window.open('');
    win.document.write('<html><head><title>Print QR Code</title></head><body style="text-align:center;padding:40px;">');
    win.document.write('<img src="' + img.src + '" style="width:300px;height:300px;"><br><br>');
    win.document.write('<button onclick="window.print();">Print</button>');
    win.document.close();
    win.focus();
}
</script>
<?php include '../includes/footer.php'; ?>
</body>
</html> 